/*
  keyboard.c
*/

#include <string.h>
#include "ram.h"
#include "keyboard.h"
#include "LCD7seg.h"
#include "keyadc.h"
#include "timer.h"
#include "display.h"
#include "hwinit.h"
#if KEY_DEBUG
#include "uart.h"
#endif

#define KEYERROR  255
#define KEYNULL   0

extern unsigned int timeBacklight;

unsigned int    rowScan;
unsigned int    lastKeyPressed;
unsigned int    nextKeyPressed;
unsigned char   doorCount, doorStatus;
#define MAX_DOOR    50//2s
unsigned char   menuCount;
#define MAX_MENU    100//4s
unsigned char   nextKeyVal;
unsigned char   nextKeyCompare[2];
byte            stopInt;
// word            bufRx[2];

struct _keyBuffer {
    unsigned char key[12];
    unsigned char *keyOut;
    unsigned char *keyIn;
} keyBuffer;

const unsigned char tabKeyRemap[TOTAL_MACHINE][MAXKEYNUM] = {
//MACHINA_AP113  
    //  0       col8    col7    col6    col5    col4    col3    col2    col1     
     {  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_0  ,  K_ENTER,K_5  ,  K_E,    // row1
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_9  ,  K_AST,  K_4  ,  K_D,    // row2
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_8  ,  K_H  ,  K_3  ,  K_C,    // row3
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_7  ,  K_G  ,  K_2  ,  K_B,    // row4
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_6  ,  K_F  ,  K_1  ,  K_A,    // row5
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row6
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row7
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row8
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL  },    

//MACHINA_NAT147
    //  0       col8    col7    col6    col5    col4    col3    col2    col1   
     {  KNULL,  KNULL,  K_F  ,  K_C  ,  K_I  ,  K_9  ,  K_3  ,  K_6  ,  K_ENTER,// row1
        KNULL,  KNULL,  K_E  ,  K_B  ,  K_H  ,  K_8  ,  K_2  ,  K_5  ,  K_0  ,  // row2
        KNULL,  KNULL,  K_D  ,  K_A  ,  K_G  ,  K_7  ,  K_1  ,  K_4  ,  K_AST,  // row3
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row4
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row5
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row6
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row7
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row8
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL  },

//MACHINA_ROWE5900
    //  0       col8    col7    col6    col5    col4    col3    col2    col1   
     {  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_1  ,  KNULL,  K_9  ,  // row1
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_5  ,  KNULL,  KNULL,  // row2
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_2  ,  KNULL,  K_0  ,  // row3
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_6  ,  KNULL,  KNULL,  // row4
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_3  ,  KNULL,  K_RESET,// row5
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_8  ,  KNULL,  KNULL,  // row6
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_4  ,  KNULL,  KNULL,  // row7
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_7  ,  KNULL,  KNULL,  // row8
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL  },

//MACHINA_ROWE6800
    //  0       col8    col7    col6    col5    col4    col3    col2    col1   
     {  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_8  ,  // row1
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_7  ,  // row2
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_6  ,  // row3
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_5  ,  // row4
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_4  ,  // row5
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_RESET,KNULL,  K_3  ,  // row6
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_0  ,  KNULL,  K_2  ,  // row7
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_9  ,  KNULL,  K_1  ,  // row8
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL  },

//MACHINA_NAT157
    //  0       col8    col7    col6    col5    col4    col3    col2    col1   
     {  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row1
        KNULL,  K_B,    K_E,    K_H,    K_2,    K_5,    K_8,    K_0,    KNULL,  // row2
        KNULL,  K_C,    K_F,    K_I,    K_3,    K_6,    K_9,    K_ENTER,KNULL,  // row3
        KNULL,  K_A,    K_D,    K_G,    K_1,    K_4,    K_7,    K_AST,  KNULL,  // row4
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row5
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row6
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row7
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row8
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL  },

//MACHINA_AP7000
    //  0       col8    col7    col6    col5    col4    col3    col2    col1   
     {  KNULL,  KNULL,  K_14 ,  K_12 ,  K_0  ,  K_8  ,  K_6  ,  K_4  ,  K_2  ,  // row1
        KNULL,  KNULL,  K_13 ,  K_11 ,  K_9  ,  K_7  ,  K_5  ,  K_3  ,  K_1  ,  // row2
        KNULL,  KNULL,  K_G  ,  K_F  ,  K_E  ,  K_D  ,  K_C  ,  K_B  ,  K_A  ,  // row3
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row4
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row5
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row6
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row7
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row8
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL  }, 

//MACHINA_LCM123
    //  0       col8    col7    col6    col5    col4    col3    col2    col1   
     {  KNULL,  K_A,    K_B,    K_C,    K_D  ,  K_E  ,  K_F  ,  K_G  ,  K_H  ,  // row1
        K_1,    K_2,    K_3,    K_4  ,  K_5  ,  K_6  ,  K_7  ,  K_8  ,  K_9  ,  // row2
        K_0  ,  K_AST,  K_ENTER,KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row3
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row4
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row5
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row6
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row7
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row8
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL  }, 

//MACHINA_AP123
    //  0       col8    col7    col6    col5    col4    col3    col2    col1   
     {  KNULL,  K_0  ,  K_7  ,  K_4  ,  K_1  ,  KNULL,  KNULL,  KNULL,  KNULL,  // row1
        KNULL,  KNULL,  K_9  ,  K_6  ,  K_3  ,  KNULL,  KNULL,  KNULL,  KNULL,  // row2
        KNULL,  K_RESET,K_8  ,  K_5  ,  K_2  ,  KNULL,  KNULL,  KNULL,  KNULL,  // row3
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row4
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row5
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row6
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row7
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row8
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL  }, 

//MACHINA_VEIDOOR
    //  0       col8    col7    col6    col5    col4    col3    col2    col1
     {  KNULL,  K_RESET,K_7  ,  K_4  ,  K_1  ,  KNULL,  KNULL,  KNULL,  KNULL,  // row 1
        KNULL,  KNULL,  K_9  ,  K_6  ,  K_3  ,  KNULL,  KNULL,  KNULL,  KNULL,  // row 2
        KNULL,  K_0  ,  K_8  ,  K_5  ,  K_2  ,  KNULL,  KNULL,  KNULL,  KNULL,  // row 3
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row 4
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row 5
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row 6
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row 7
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row 8
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL  },

//MACHINA_NAT145
    //  0       col8    col7    col6    col5    col4    col3    col2    col1   
     {  KNULL,  KNULL,  K_7  ,  K_6  ,  K_8  ,  K_AST,  K_9  ,  K_0  ,  K_ENTER,// row1
        KNULL,  KNULL,  K_2  ,  K_1  ,  K_3  ,  K_13 ,  K_4  ,  K_5  ,  K_14 ,  // row2
        KNULL,  KNULL,  K_B  ,  K_A  ,  K_C  ,  K_F  ,  K_D  ,  K_E  ,  K_G,    // row3
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row4
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row5
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row6
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row7
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row8
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL  }, 
#if TOK_TAKE
//MACHINA_AMS39 per TOK_TAKE
    //  0       col8    col7    col6    col5    col4    col3    col2    col1   
     {  KNULL,  KNULL,  KNULL,  KNULL,  K_1,    K_4,    K_7,    K_AST,  KNULL,  // row1
        KNULL,  KNULL,  KNULL,  KNULL,  K_2,    K_5,    K_8,    K_0,    KNULL,  // row2
        KNULL,  KNULL,  KNULL,  KNULL,  K_3,    K_6,    K_9,    K_RESET,KNULL,  // row3
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row4
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row5
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row6
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row7
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row8
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL  },
#else
//MACHINA_AMS39
    //  0       col8    col7    col6    col5    col4    col3    col2    col1   
     {  KNULL,  KNULL,  KNULL,  KNULL,  K_0  ,  K_8  ,  K_6  ,  K_4  ,  K_2  ,  // row1
        KNULL,  KNULL,  KNULL,  KNULL,  K_9  ,  K_7  ,  K_5  ,  K_3  ,  K_1  ,  // row2
        KNULL,  KNULL,  KNULL,  KNULL,  K_RESET,KNULL,  KNULL,  KNULL,  KNULL,  // row3
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row4
        KNULL,  KNULL,  KNULL,  KNULL,  K_AST,  KNULL,  KNULL,  KNULL,  KNULL,  // row5
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row6
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row7
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row8
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL  },
#endif
//MACHINA_VEI147
    //  0       col8    col7    col6    col5    col4    col3    col2    col1   
     {  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row1
        KNULL,  KNULL,  KNULL,  KNULL,  K_2,    K_5,    K_8,    K_0,    KNULL,  // row2
        KNULL,  KNULL,  KNULL,  KNULL,  K_3,    K_6,    K_9,    K_ENTER,KNULL,  // row3
        KNULL,  KNULL,  KNULL,  KNULL,  K_1,    K_4,    K_7,    K_RESET,KNULL,  // row4
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row5
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row6
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row7
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row8
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL  },

//MACHINA_USIGVC2
    //  0       col8    col7    col6    col5    col4    col3    col2    col1   
     {  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_3,    K_2,    K_1,    // row1
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_6,    K_5,    K_4,    // row2
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_9,    K_8,    K_7,    // row3
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row4
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_ENTER,K_0,    K_RESET,// row5
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row6
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row7
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row8
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL  },

//MACHINA_MERCHA6
    //  0       col8    col7    col6    col5    col4    col3    col2    col1     
     {  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_RESET,K_7  ,  K_4  ,  K_1  ,  // row1
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_ENTER,K_9  ,  K_6  ,  K_3  ,  // row2 
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  K_0  ,  K_8  ,  K_5  ,  K_2  ,  // row3
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row4 
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row5
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row6
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row7
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row8
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL  },
        
//MACHINA_VEILCM
    //  0       col8    col7    col6    col5    col4    col3    col2    col1
     {  KNULL,  K_RESET,K_7  ,  K_4  ,  K_1  ,  KNULL,  KNULL,  KNULL,  KNULL,  // row 1
        KNULL,  KNULL,  K_9  ,  K_6  ,  K_3  ,  KNULL,  KNULL,  KNULL,  KNULL,  // row 2
        KNULL,  K_0  ,  K_8  ,  K_5  ,  K_2  ,  KNULL,  KNULL,  KNULL,  KNULL,  // row 3
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row 4
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row 5
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row 6
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row 7
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  // row 8
        KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL,  KNULL  },

};

const unsigned char tabKeyToAscii[] =
{
   0  ,
   'A',       //  K_PLU1
   'B',       //  K_PLU2
   'C',       //  K_PLU3
   'D',       //  K_PLU4
   'E',       //  K_PLU5
   'F',       //  K_PLU6
   'G',       //  K_PLU7
   'H',       //  K_PLU8
   'I',       //  K_PLU9
   'J',       //  K_PLU10
   'K',       //  K_PLU11
   'L',       //  K_PLU12
   'M',       //  K_PLU13
   'N',       //  K_PLU14
   'O',       //  K_PLU15
   'P',       //  K_PLU16
   'Q',       //  K_PLU17
   'R',       //  K_PLU18
   'S',       //  K_PLU19
   'T',       //  K_PLU20
   'U',       //  K_PLU21
   'V',       //  K_PLU22
   'W',       //  K_PLU23
   'X',       //  K_PLU24
   'Y',       //  K_PLU25
   'Z',       //  K_PLU26
   ' ',       //  K_PLU27
   ',',       //  K_PLU28
   '%',       //  K_PLU29
   0,         //  K_PLU30
   0,         //  K_PLU31
   0,         //  K_PLU32
   0,         //  K_PLU33
   0,         //  K_PLU34
   0,         //  K_PLU35 (euro)
   0,
   0,
   0,
   0,
   '0',       //  K_0
   '1',       //  K_1
   '2',       //  K_2
   '3',       //  K_3
   '4',       //  K_4
   '5',       //  K_5
   '6',       //  K_6
   '7',       //  K_7
   '8',       //  K_8
   '9',       //  K_9
   0  ,       //  K_00
   0  ,       //  K_V1
   0  ,       //  K_V2
   0  ,       //  K_V3
   0  ,       //  K_V4
   0  ,       //  K_V5
   0  ,       //  K_V6
   0,
   0,
   0,
   '+',       //  K_PIU
   '-',       //  K_MENO
   'X',       //  K_PER
   0  ,       //  K_PLU
   0  ,       //  K_C
   0  ,       //  K_TARAUTO
   0  ,       //  K_TARAMAN
   0  ,       //  K_ZERO
   0  ,       //  K_CARTA1
   0  ,       //  K_CARTA2
   0  ,       //  K_TOT
   0  ,       //  K_R
   0  ,       //  K_KEY
   0  ,       //  K_ENTER
   0  ,       //  K_LOCK
};




/*--------------------------------------------------------------------------
 | keyStart:    start keyboard buffers
 |                              --------------
 | In:
 | Out:
 +--------------------------------------------------------------------------*/

void            keyStart( void )
{
    stopInt = 1;
    memset( &ADkey, 0, sizeof(ADkey) );
    memset( &keyBuffer, 0, sizeof(keyBuffer) );
    keyBuffer.keyOut = keyBuffer.key;
    keyBuffer.keyIn = keyBuffer.key;
    nextKeyCompare[0] = nextKeyCompare[1] = 0;
    doorCount = 0;
    doorStatus = FALSE;
    menuCount = 0;
    rowScan = 0;
    lastKeyPressed = 0;
    stopInt = 0;
}                                               //  keyStart




/*--------------------------------------------------------------------------
 | keyFlush:    flush keyboard buffers
 |                              --------------
 | In:
 | Out:
 +--------------------------------------------------------------------------*/

void            keyFlush( void )
{
    stopInt = 1;
    memset( &keyBuffer, 0, sizeof(keyBuffer) );
    keyBuffer.keyOut = keyBuffer.key;
    keyBuffer.keyIn = keyBuffer.key;
    stopInt = 0;
}                                               //  keyFlush




/*--------------------------------------------------------------------------
 | doorOpen:  restituisce lo stato del sensore porta aperta
 |                              --------------
 | In:
 | Out: TRUE    porta aperta
 |      FALSE   porta chiusa
 +--------------------------------------------------------------------------*/

#ifndef __WIN32__
int             doorOpen( void )
{
    return doorStatus;
}                                               //  doorOpen
#endif //__WIN32__

/*--------------------------------------------------------------------------
 | MenuSwitch:  restituisce lo stato del pulsante per passare in programmazione
 |                              --------------
 | In:
 | Out: TRUE    porta aperta
 |      FALSE   porta chiusa
 +--------------------------------------------------------------------------*/

#ifndef __WIN32__
int             MenuSwitch( void )
{
    return (menuCount==MAX_MENU);
}                                               //  MenuSwitch
#endif //__WIN32__


/*--------------------------------------------------------------------------
 | kbhit:       restituisce il numero di tasti premuti
 |                              --------------
 | In:
 | Out:
 +--------------------------------------------------------------------------*/

int             kbhit( void )
{
#ifdef  __WIN32__
    Sleep(0);
#endif
    if ( keyBuffer.keyIn >= keyBuffer.keyOut )
    {
        return( keyBuffer.keyIn - keyBuffer.keyOut );
    }
    else
    {
        return( sizeof(keyBuffer) + keyBuffer.keyIn - keyBuffer.keyOut );
    }
}                                               //  kbhit




/*--------------------------------------------------------------------------
 | getKey:      toglie un tasto dal buffer
 |                              --------------
 | In:
 | Out: byte    codice del tasto letto
 +--------------------------------------------------------------------------*/
byte            getKey( void )
{
    byte        key;

#ifdef  __WIN32__
    Sleep(0);
#endif

    key = 0;
    if ( keyBuffer.keyOut != keyBuffer.keyIn )
    {
        key = *keyBuffer.keyOut;
        if ( ++keyBuffer.keyOut >= (keyBuffer.key+sizeof(keyBuffer.key)) )
            keyBuffer.keyOut = keyBuffer.key;
    }
    if( machineType == MACHINA_NAT145   || machineType == MACHINA_NAT147 || machineType == MACHINA_NAT157 || 
        machineType == MACHINA_ROWE6800 || machineType == MACHINA_AP7000 || machineType == MACHINA_AP123 || 
        machineType == MACHINA_VEIDOOR  || machineType == MACHINA_AMS39 || machineType == MACHINA_VEI147 ||
        machineType == MACHINA_USIGVC2  || machineType == MACHINA_MERCHA6 || machineType == MACHINA_VEILCM ||
        machineType == MACHINA_AP113   )        // rev 7.3.6
      key = tabKeyRemap[machineType][key];
    return key;
}                                               //  getKey




/*--------------------------------------------------------------------------
 | putKey:      inserisce nel buffer il tasto letto ad interrupt
 |                              --------------
 | In:  _key    codice del tasto
 | Out:
 +--------------------------------------------------------------------------*/

void            putKey( byte _key )
{
    if ( stopInt )
        return;

    if ( keyBuffer.keyIn >= keyBuffer.keyOut )
    {
      if( ( keyBuffer.keyIn - keyBuffer.keyOut ) >= sizeof(keyBuffer.key)-1 )
          return;
    }
    else
    {
      if( ( keyBuffer.keyOut - keyBuffer.keyIn ) <= 1 )
          return;
    }

    lampOn();
    BuzzerOn(50);
    *keyBuffer.keyIn = _key;
    if ( ++keyBuffer.keyIn >= (keyBuffer.key+sizeof(keyBuffer.key)) )
        keyBuffer.keyIn = keyBuffer.key;
}                                               //  putKey




/*--------------------------------------------------------------------------
 | ISRkey:      interrupt routine
 |                              --------------
 | In:
 | Out:
 +--------------------------------------------------------------------------*/

#define KEY_AN(a)  a

//the 2 wire keyboard and the matrix keyboard can be interchangeable
//tabANkey[] is now used to map the analog value received by the 2 wire keyboard 
//to the position of tabKeyRemap[] array where find kay value
//KEY_AN(936) was coupled with key K_9 now KEY_AN(936) is coupled with value 14
//which is the position of tabKeyRemap[] where find K_9

//const struct _TAB_KEY_ {
//    word        value;
//    byte        key;
//} tabANkey[] = {
//    {   KEY_AN(1000),    KNULL   },
//    {   KEY_AN(967),    K_0     },//967
//    {   KEY_AN(936),    K_9     },
//    {   KEY_AN(903),    K_8     },
//    {   KEY_AN(868),    K_7     },
//    {   KEY_AN(832),    K_6     },
//    {   KEY_AN(790),    K_5     },
//    {   KEY_AN(754),    K_4     },
//    {   KEY_AN(712),    K_3     },
//    {   KEY_AN(668),    K_2     },
//    {   KEY_AN(622),    K_1     },
//    {   KEY_AN(574),    K_ENTER },
//    {   KEY_AN(531),    K_AST   },
//    {   KEY_AN(484),    K_H     },
//    {   KEY_AN(433),    K_G     },
//    {   KEY_AN(379),    K_F     },
//    {   KEY_AN(324),    K_E     },
//    {   KEY_AN(266),    K_D     },
//    {   KEY_AN(205),    K_C     },
//    {   KEY_AN(139),    K_B     },
//    {   KEY_AN( 73),    K_A     },
//    {   KEY_AN(  0),    KNULL   },
//    {   KEY_AN(  0),    KNULL   },
//    {   KEY_AN(  0),    KNULL   },
//},
const struct _TAB_KEY_ {
    word        value;
    byte        key;
} tabANkey[] = {
    {   KEY_AN(1000),   KNULL   },
    {   KEY_AN(967),      5     },
    {   KEY_AN(936),     14     },
    {   KEY_AN(903),     23     },
    {   KEY_AN(868),     32     },
    {   KEY_AN(832),     41     },
    {   KEY_AN(790),      7     },
    {   KEY_AN(754),     16     },
    {   KEY_AN(712),     25     },
    {   KEY_AN(668),     34     },
    {   KEY_AN(622),     43     },
    {   KEY_AN(574),      6     },
    {   KEY_AN(531),     15     },
    {   KEY_AN(484),     24     },
    {   KEY_AN(433),     33     },
    {   KEY_AN(379),     42     },
    {   KEY_AN(324),      8     },
    {   KEY_AN(266),     17     },
    {   KEY_AN(205),     26     },
    {   KEY_AN(139),     35     },
    {   KEY_AN( 73),     44     },
    {   KEY_AN(  0),    KNULL   },
    {   KEY_AN(  0),    KNULL   },
    {   KEY_AN(  0),    KNULL   },
},
tabANkeyROWE[] = {
    {   KEY_AN(999),    KNULL   },
    {   KEY_AN(898),    K_9     },
    {   KEY_AN(786),    K_0     },
    {   KEY_AN(673),    K_RESET },
    {   KEY_AN(611),    K_1     },
    {   KEY_AN(556),    K_2     },
    {   KEY_AN(500),    K_3     },
    {   KEY_AN(433),    K_8     },
    {   KEY_AN(363),    K_4     },
    {   KEY_AN(286),    K_7     },
    {   KEY_AN(200),    K_6     },
    {   KEY_AN(106),    K_5     },
    {   KEY_AN(  0),    KNULL   }
};

void            filterKey( void )
{
    word        mask;

    mask = AD.conv[AD_2];

    if ( machineType != MACHINA_AP113 )
    {
        ADkey.totMax -= ADkey.valMax[ADkey.iMax];
        ADkey.valMax[ADkey.iMax] = mask;
        ADkey.totMax += ADkey.valMax[ADkey.iMax];
        if ( ADkey.iMax >= 127 )
            ADkey.iMax = 0;
        else
            ADkey.iMax++;
    }
}

void            ISRkey( void )
{
#ifndef __WIN32__
 #if !RSR903
    word        mask;
    BOOL        flag;
    byte        i;
    byte        keyInData, keyInMenu, keyInDoor;
    struct _TAB_KEY_ const *p;

    if ( machineType == MACHINA_LCM123 )
    {
        i = 0;
        keyInMenu = inImage[2] | 0x8E;
        keyInData = inImage[3] | 0x80;
        if ( keyInMenu == 191 )
        {
            if ( keyInData == 254 )         i = K_A;
        }
        else if ( keyInMenu == 254 )
        {
            if ( keyInData == 254 )         i = K_B;
            else if ( keyInData == 191 )    i = K_AST;
        }
        else if ( keyInMenu == 255 )
        {
            if ( keyInData == 190 )         i = K_C;
            else if ( keyInData == 187 )    i = K_D;
            else if ( keyInData == 183 )    i = K_E;
            else if ( keyInData == 215 )    i = K_F;
            else if ( keyInData == 243 )    i = K_G;
            else if ( keyInData == 249 )    i = K_3;
            else if ( keyInData == 235 )    i = K_5;
            else if ( keyInData == 207 )    i = K_7;
            else if ( keyInData == 159 )    i = K_9;
        }
        else if ( keyInMenu == 190 )
        {
            if ( keyInData == 255 )         i = K_H;
        }
        else if ( keyInMenu == 239 )
        {
            if ( keyInData == 253 )         i = K_1;
            else if ( keyInData == 251 )    i = K_2;
        }
        else if ( keyInMenu == 223 )
        {
            if ( keyInData == 251 )         i = K_4;
            else if ( keyInData == 239 )    i = K_6;
            else if ( keyInData == 223 )    i = K_8;
            else if ( keyInData == 191 )    i = K_0;
        }
        else if ( keyInMenu == 222 )
        {
            if ( keyInData == 255 )         i = K_ENTER;
        }
        nextKeyPressed = i;
    }
    
                                        // legge tastiera analogica AP113
    if ( machineType == MACHINA_AP113 )
    {
        mask = AD.conv[AD_2];
#if KEY_DEBUG
        commPutChar( COMM1, (mask>>2) );
#endif
        {
            if ( mask >= (ADkey.lastAn-2) && mask <= (ADkey.lastAn+2) )
            {//lettura stabile entro +/-2
                if ( ADkey.nCamp == 32 )
                {//per 32 letture consecutive
                    if ( mask > (ADkey.valMax[1]+4) || mask < (ADkey.valMax[1]-4) )
                    {// update values only if different from previous
                        ADkey.valMax[0] = ADkey.valMax[1];  // update old value
                        ADkey.valMax[1] = mask;             // update new value
                        if ( mask < ADkey.valMax[0] )
                        {                                   // if less than previous set max to last one
                            ADkey.maxAn = ADkey.valMax[0];
                            mask = mask;                    // and key to new one
                            ADkey.nCamp = 0;
                        }
                        else
                        {
                            ADkey.maxAn = mask;
                        }

                        if ( ADkey.maxAn > (mask+6) )
                        {
                            if ( ADkey.maxAn )
                                mask = (mask*1000)/ADkey.maxAn;
                            p = tabANkey;
                            for ( i = 0; tabANkey[i].value != 0; i++, p++ )
                            {
//                                if ( testMode == 0x47681022 )
//                                    flag =  mask <=   ( ((p+0)->value)+  ((p+1)->value)*2 )/3 &&
//                                            mask >=   ( ((p+2)->value)+  ((p+1)->value)*2 )/3;
//                                else
                                    flag =  mask <=   ( ((p+0)->value)+  ((p+1)->value) )/2 &&
                                            mask >=   ( ((p+1)->value)+  ((p+2)->value) )/2;
                                if ( flag )
                                {
                                    mask = tabANkey[i+1].key;
                                    if ( mask == ADkey.lastKey )
                                        break;
                                    ADkey.lastKey = mask;
                                    putKey(mask);
#if KEY_DEBUG
        commPutChar( COMM1, 0 );
#endif
                                    break;
                                }
                            }
                        }
                    }
                }
                else
                    ADkey.nCamp++;
            }
            else
            {
                if ( ADkey.nCamp == 0 )
                {
                    ADkey.lastAn = mask;
                    ADkey.lastKey = 0;
                }
                else
                    ADkey.nCamp--;
            }
        }
    }
                                        // legge tastiera analogica ROWE5900
    if ( machineType == MACHINA_ROWE5900 )
    {
        mask = AD.conv[AD_2];

        if ( (ADkey.totMax)/128 < 50 )
        {                                       // aggiorna massimo analogico (nessun tasto)
            ADkey.nCamp = 0;
            ADkey.lastAn = 0;
        }
        else
        {
            if ( mask >= (ADkey.lastAn-4) && mask <= (ADkey.lastAn+4) )
            {
                if ( ADkey.nCamp == 32 || ADkey.nCamp == 255 )
                {
                    for ( i = 0; tabANkeyROWE[i].value != 0; i++ )
                    {
                        if ( testMode == 0x47681022 )
                            flag =  mask <= (tabANkeyROWE[i  ].value+tabANkeyROWE[i+1].value*2)/3 &&
                                    mask >= (tabANkeyROWE[i+2].value+tabANkeyROWE[i+1].value*2)/3;
                        else
                            flag =  mask <= (tabANkeyROWE[i  ].value+tabANkeyROWE[i+1].value)/2 &&
                                    mask >= (tabANkeyROWE[i+1].value+tabANkeyROWE[i+2].value)/2;
                        if ( flag )
                        {
                            mask = tabANkeyROWE[i+1].key;
                            ADkey.nCamp = 255;
                            if ( mask == ADkey.lastKey )
                                break;
                            ADkey.lastKey = mask;
                            putKey(mask);
                            break;
                        }
                    }
                }
                else
                    ADkey.nCamp++;
            }
            else
            {
                if( ADkey.nCamp == 0 )
                {
                    ADkey.lastAn = mask;
                    ADkey.lastKey = 0;
                }
                else
                    ADkey.nCamp--;
            }
        }
    }

    flag = 0;
                                        // legge tastiera in scansione
    if ( machineType == MACHINA_ROWE5900 || machineType == MACHINA_ROWE6800 )
    {
        keyInData = inImage[3] | 0xFA;
        keyInMenu = INP0;
        keyInDoor = inImage[3] & 0x40;
    }
    else if ( machineType == MACHINA_AP113 || machineType == MACHINA_AP7000 )
    {
        keyInData = inImage[3] | 0x80;
        keyInMenu = inImage[2] & 0x10;
        keyInDoor = (inImage[2] & 0x40)?0:1;
        if ( machineType == MACHINA_AP7000 )
            keyInDoor ^= 0x01;
    }
    else if ( machineType == MACHINA_NAT157 || machineType == MACHINA_VEI147 )
    {
        keyInData = inImage[3];
        keyInMenu = (AD.conv[AD_2] > 700 && AD.conv[AD_2] < 850) || AD.conv[AD_2] < 150;
        keyInDoor = AD.conv[AD_2] > 650;//509...771...924

        if ( rowScan == 4 )
            flag = 1;
    }
    else if ( machineType == MACHINA_LCM123 )
    {
        keyInMenu = (AD.conv[AD_2] > 700 && AD.conv[AD_2] < 850) || AD.conv[AD_2] < 150;
        keyInDoor = AD.conv[AD_2] > 650;//509...771...924
        flag = 1;
    }
    else if ( machineType == MACHINA_AP123 || machineType == MACHINA_VEIDOOR || machineType == MACHINA_VEILCM )
    {
        keyInData = inImage[3] | 0x03;
        keyInMenu = (inImage[3] & 0x03)^0x03;
        keyInDoor = (inImage[2] & 0x40)?0:1;
        flag = 1;
    }
    else if ( machineType == MACHINA_AMS39 || machineType == MACHINA_USIGVC2 )
    {
        keyInData = inImage[3] | 0xE0;
        keyInMenu = (inImage[3] & 0x40);
        keyInDoor = (inImage[2] & 0x40)?0:1;
    }
    else if ( machineType == MACHINA_MERCHA6 )
    {
        keyInData = inImage[3] | 0xF0;
        keyInMenu = (inImage[3] & 0x40);
        keyInDoor = (inImage[2] & 0x40)?1:0;
    }   
    else
    {
        keyInData = inImage[3] | 0x80;
        keyInMenu = inImage[3] & 0x40;
        keyInDoor = inImage[2] & 0x20;
    }
    if ( flag )
    {                                   // check menu switch
        if ( keyInMenu != 0 )
        {
            if ( menuCount < MAX_MENU )
                menuCount++;
        }
        else
        {
            if ( menuCount > 0 )
                menuCount--;
        }
    }

    if ( ++keyTimer._1ms >= 32 )
    {
        keyTimer._32ms++;
        keyTimer._1ms = 0;
        if ( keyTimer._32ms >= 2 )
            keyTimer._32ms = 0;
    }

    // la lettura della tastiera e' codificata in bufKeyInt con:
    // da 0 a 7 column 6 7 8 4 3 2 1 9, la 8 contiene column5 (bit0 row1...)
    // keyTimer._1ms 0...1...2...3...4...5...6...7...8...9..10..11..31..0
    // rowScan       000000001111222233334444555566667777...............0
    // keyInData     xxxxxxxxxxxx00001111222233334444555566667777xxxxxxxx
    if ( keyTimer._1ms == 0 )
    {
        rowScan = 0;
        KEY_MUX(rowScan);
    }
    else if ( keyTimer._1ms == 1 )
    {
        nextKeyPressed = 0;
        nextKeyVal = 1;                 // il numero del tasto premuto parte da 1
    }
    else if ( keyTimer._1ms <= 10 && machineType != MACHINA_LCM123 )
    {
        if ( keyTimer._1ms >= 3 && rowScan <= 8 )
        {
            if ( rowScan == 4 && machineType != MACHINA_NAT157 && machineType != MACHINA_VEI147 )
            {                           // pulsante menu
                if ( keyInMenu == 0x000 )
                {
                    if ( menuCount < MAX_MENU )
                        menuCount++;
                }
                else
                {
                    if ( menuCount > 0 )
                        menuCount--;
                }
                keyInData |= 0x040;
            }

                                        // tastiera
            if ( keyInData != 0x00FF )
            {
                if ( nextKeyPressed == 0 )
                {                       // primo tasto premuto
                    for( mask = 0x0080; (mask & keyInData) != 0; mask >>= 1, nextKeyVal++ )
                        ;
                    nextKeyPressed = nextKeyVal;
                }
                else   // secondo tasto premuto
                    nextKeyPressed = KEYERROR;
            }
            else
                nextKeyVal += 9;
        }

        rowScan++;
        KEY_MUX(rowScan);
    }
    else if ( keyTimer._1ms == 11 )
    {
        nextKeyCompare[keyTimer._32ms] = nextKeyPressed;
        if ( nextKeyCompare[0] == nextKeyCompare[1] && lastKeyPressed != nextKeyCompare[0] && nextKeyCompare[0] != KEYERROR )
        {
            lastKeyPressed = nextKeyCompare[0];
            if ( nextKeyCompare[0] != KEYNULL )
                putKey(lastKeyPressed);
        }

        if ( keyInDoor != 0x000 )
        {
            if ( doorCount < MAX_DOOR )
                doorCount++;
            else
                doorStatus = TRUE;
        }
        else            
        {
            if ( doorCount > 0 )
                doorCount--;
            else
                doorStatus = FALSE;
        }
    }
 #else    
    if ( IO1PIN_bit.P1_27 != 0x00 )
    {
        if ( doorCount < MAX_DOOR )
            doorCount++;
        else
            doorStatus = TRUE;
//mmm                        doorStatus = FALSE;
    }
    else
    {
        if ( doorCount > 0 )
            doorCount--;
        else
            doorStatus = FALSE;
    }  
 #endif
#endif
}                                               //  ISRkey
