/******************************************************************************
hwinit.c
init hw scheda
*******************************************************************************/

#include	<string.h>

#include "ram.h"
#include "lcd7seg.h"
#include "hwinit.h"
#include "def.h"
#include "timer.h"
#ifdef __WIN32__
#include <time.h>
#else
#include "LPC_vic.h"
#endif
#include "keyadc.H"
#include "display.h"
#include "inizial.h"
#include "keyboard.h"
#include "main.h"
#include "messaggi.h"
#include "uart.h"
#include "ds2341.h"
#include "test.h"
#include "device.h"


#define  PLLFEED_DATA1     0xAA
#define  PLLFEED_DATA2     0x55

extern unsigned char bufLcdIIC[20];

const unsigned char dividerVpb[] = { 0, 1, 2, 0, 0 };




/*--------------------------------------------------------------------------
 | HardwareInit:hardware init ad alto livello
 |                              --------------
 | In:
 | Out:
 +--------------------------------------------------------------------------*/

void            HardwareInit( void )
{

#ifndef __WIN32__
    __disable_interrupt();
#endif

    InitPLL();
#ifndef BOOT
#if BOARDTEST
    testCPU();
#endif
#endif
    InitIOPorts();
    InitInterrupt();
    LcdIIC0Init();
    initSPI1();
    AdcCpuInit();
    initSPI0();
    initSSP0();
    start_up_and_reset(); // usa SPI1

    InitTimer0();
    InitTimer1();
    InitTimer2();
    InitPwmTimer();

//    start_up_and_reset();

#ifdef __WIN32__
    machineType = MACHINA_NAT147;
#else
    rsr903c_detection();    
    checkMachineType();   
#endif

#ifndef __WIN32__
    __enable_interrupt();
#endif

    setContrastLCD(1);
    contrastInit();
    LcdErase(TUTTI);
    Delay(16);                          // vcc stabile
    DispInit();
    DispClrScr();
}                                               //  HardwareInit




/*--------------------------------------------------------------------------
 | InitPLL:     inizializzazione PLL
 |                  CCLK = Fcco / (2 * P)
 |                  cclk -> pll out freq
 |                  Fcco -> freq controlled oscillator
 |                  P -> pll divider (PSEL)
 |                  dove Fcco = Fosc * M * 2 * P
 |                  M -> pll multiplier (MSEL)
 |                  per calcolo di nuovi valori riferirsi al foglio excel allegato
 |                  Fout = 60MHz
 |                              --------------
 | In:
 | Out:
 +--------------------------------------------------------------------------*/

#define PLL_MValue	       11
#define PLL_NValue		0
#define CCLKDivValue		5

/* System configuration: Fosc, Fcclk, Fcco, Fpclk must be defined */
/* PLL input Crystal frequence range 4KHz~20MHz. */
#define Fosc	 12000000
/* System frequence,should be less than 80MHz. */
#define Fcco	(2*Fosc*(PLL_MValue+1))/(PLL_NValue+1)  //288000000
#define Fcclk	Fcco/(CCLKDivValue+1)                   // 48000000

void            InitPLL( void )
{
#ifndef __WIN32__


{
	DWORD MValue, NValue;

    SCS &= ~BIT3;
    
    if ( PLLSTAT & (1 << 25) )
    {
		PLLCON = 1;			/* Enable PLL, disconnected */
		PLLFEED = PLLFEED_DATA1;
		PLLFEED = PLLFEED_DATA2;
    }

    PLLCON = 0;				/* Disable PLL, disconnected */
    PLLFEED = PLLFEED_DATA1;
    PLLFEED = PLLFEED_DATA2;

	SCS |= 0x20;			/* Enable main OSC */
    while( !(SCS & 0x40) );	/* Wait until main OSC is usable */

    CLKSRCSEL = 0x1;		/* select main OSC, 12MHz, as the PLL clock source */

    PLLCFG = PLL_MValue | (PLL_NValue << 16);
    PLLFEED = PLLFEED_DATA1;
    PLLFEED = PLLFEED_DATA2;

    PLLCON = 1;				/* Enable PLL, disconnected */
    PLLFEED = PLLFEED_DATA1;
    PLLFEED = PLLFEED_DATA2;

    CCLKCFG = CCLKDivValue;	/* Set clock divider */

    while ( ((PLLSTAT & (1 << 26)) == 0) );	/* Check lock bit status */

    MValue = PLLSTAT & 0x00007FFF;
    NValue = (PLLSTAT & 0x00FF0000) >> 16;
    while ((MValue != PLL_MValue) && ( NValue != PLL_NValue) );

    PLLCON = 3;				/* enable and connect */
    PLLFEED = PLLFEED_DATA1;
    PLLFEED = PLLFEED_DATA2;
	while ( ((PLLSTAT & (1 << 25)) == 0) );	/* Check connect bit status */
}
/*
#if 0
    SCS = 0x00;
    CLKSRCSEL = 0x00;                   // internal oscillator
#else
    SCS = 0x20;
    while( (SCS & 0x40) == 0x00 );


    CLKSRCSEL = 0x00;                   // internal oscillator
//    for( i = 0; i < 50000; i++ );

    PLLCON_bit.PLLC = 0;	            // disconnect PLL
    PLLFEED = PLLFEED_DATA1;
    PLLFEED = PLLFEED_DATA2;

//    for( i = 0; i < 50000; i++ );

    PLLCON_bit.PLLE = 0;                // disable PLL
    PLLFEED = PLLFEED_DATA1;
    PLLFEED = PLLFEED_DATA2;

//    for( i = 0; i < 50000; i++ );

#if 1
// Fcco = (2 * M * Fosc ) * N = (2 * (M=72) * 4MHz ) / (N=2)
// CCLK = Fcco / (P+1) = 288MHz / (1+4) = 57,6MHz
    CCLKCFG   = 0x04;                   // CPU clock divide by 5

    PLLCFG = 0x00010048;                // 00000000-00000001-00000000-00011111
                                        // xxxxxxxx                             reserved
                                        //          00000001                    NSEL
                                        //                   x                  reserved
                                        //                    0000000-00011111  MSEL
#else
// Fcco = (2 * M * Fosc ) * N = (2 * (M=32) * 12MHz ) / (N=2)
// CCLK = Fcco / (P+1) = 384MHz / (1+5) = 64MHz
    CCLKCFG   = 0x05;                   // CPU clock divide by 6

//    CLKSRCSEL = 0x01;                   // select main oscillator

    PLLCFG = 0x0001001F;                // 00000000-00000001-00000000-00011111
                                        // xxxxxxxx                             reserved
                                        //          00000001                    NSEL
                                        //                   x                  reserved
                                        //                    0000000-00011111  MSEL
#endif

    PLLCON_bit.PLLE = 1;                // Enable PLL
    PLLFEED = PLLFEED_DATA1;
    PLLFEED = PLLFEED_DATA2;
    while (!PLLSTAT_bit.PLOCK);         // Wait PLL lock
    PLLCON_bit.PLLC = 1;	            // Connect PLL
    PLLFEED = PLLFEED_DATA1;
    PLLFEED = PLLFEED_DATA2;
#endif///0
*/

    PCONP = 0x7FFFFFFF;                 // accende tutto tranne USB
    PCLKSEL0 = 0x00000000;              // CCLK/4
    PCLKSEL1 = 0x00000000;

   //EEE    VPBDIV_bit.VPBDIV = dividerVpb[VPB_DIVIDER];  // perif clock  
#ifdef BOOT2
    memcpy((void*)0x40000000, (void const*)0x8000, 0x40);
    MEMMAP_bit.MAP = USER_RAM;        // Do MEMAMP 
#else    
    MEMMAP_bit.MAP = USER_FLASH;        // Do MEMAMP   
#endif    
    MAMCR = 0;
    MAMTIM = 3;
    MAMCR = 1;          // FAIL MAM.1 20.07.2007; misure su interrupt timer MAM=2 43uS, MAM=1 46uS

#endif
}                                               //  InitPLL




/*--------------------------------------------------------------------------
 | InitIOPorts: inizializza pin di I/O del micro
 |                              --------------
 | In:
 | Out:
 +--------------------------------------------------------------------------*/

void            InitIOPorts( void )
{
#ifndef __WIN32__

    PINSEL10= 0x00000000;               // 00000000 00000000 00000000 00000000


    PINSEL1 = 0x01652A81;               // 00000001
                                        // xx         31 nc
                                        //   xx       30 port
                                        //     xx     29 port
                                        //       xx   28 scl0
                                        // 01100101
                                        // xx         27 sda0
                                        //   xx       26 aout
                                        //     xx     25 ad0.2 : tastiera esterna
                                        //       xx   24 ad0.1 : pwr mon
                                        // 00101010
                                        // xx         23 ad0.0 : imotor
                                        //   xx       22 mcidat0
                                        //     xx     21 mcipwr
                                        //       xx   20 mcicmd
                                        // 10000001
                                        // xx         19 mciclk
                                        //   xx       18 port                   CTS maxstream
                                        //     xx     17 port                   RTS maxstream
                                        //       xx   16 rxd1

    PINSEL0 = 0x405A8050;               // 01000000
                                        // xx         15 txd1
                                        //   xx       14 nc
                                        //     xx     13 nc
                                        //       xx   12 nc
                                        // 01011010
                                        // xx         11 rxd2
                                        //   xx       10 txd2
                                        //     xx     09 spi1 mosi
                                        //       xx   08 spi1 miso
                                        // 10000000
                                        // xx         07 spi sck
                                        //   xx       06 port                   spi0 mosi key
                                        //     xx     05 port                   spi0 miso key
                                        //       xx   04 port                   spi0 sck key
                                        // 01010000
                                        // xx         03 rxd0
                                        //   xx       02 txd0
                                        //     xx     01 port                   iic1 sck
                                        //       xx   00 port                   iic1 sda
                                        // NOTE RSR GPIO0.30 diventa uscita solo se IODIR_P0_29 e IODIR_P0_30 sono entrambi ad 11
    IO0CLR  = 0xF3FFFF5F;               // 11110011 11111111 11111111 01011111
    IO0SET  = 0x000000A0;               // 00000000 00000000 00000000 10100000
    IO0DIR  = 0x7C7885F0;               // 01111100 01111000 10000101 11110000


    PINSEL3 = 0x0003C305;               // 00000000
                                        // xx         31 port
                                        //   xx       30 port
                                        //     xx     29 port
                                        //       xx   28 port
                                        // 00000011
                                        // xx         27 port
                                        //   xx       26 led 2
                                        //     xx     25 led 1
                                        //       xx   24 mosi0
                                        // 11000011
                                        // xx         23 miso0
                                        //   xx       22 port-out
                                        //     xx     21 port-out
                                        //       xx   20 sck0
                                        // 00000101
                                        // xx         19 inp1
                                        //   xx       18 inp0
                                        //     xx     17 enet mdio
                                        //       xx   16 enet mdc

//    PINSEL2 = 0x50150105;
    PINSEL2 = 0x00000000;               // 01010000
                                        // xx         15 enet ref clk
                                        //   xx       14 enet rx er
                                        //     xx     13 nc
                                        //       xx   12 nc
                                        // 00010101
                                        // xx         11 nc
                                        //   xx       10 enet rxd1
                                        //     xx     09 enet rdx0
                                        //       xx   08 enet crs
                                        // 00000001
                                        // xx         07 nc
                                        //   xx       06 nc
                                        //     xx     05 nc
                                        //       xx   04 enet txen
                                        // 00000101
                                        // xx         03 nc
                                        //   xx       02 nc
                                        //     xx     01 enet txd1
                                        //       xx   00 enet txd0

#if ANTI_V
    IO1DIR  = 0x07500000;               // 00000110 01000000 00000000 00000000 ???
#else
    IO1DIR  = 0x07500000;               // 00000111 01010000 00000000 00000000
#endif
    IO1SET  = 0x06000000;               //
    IO1CLR  = 0xF9FFFFFF;               //
    
#if LED_DIMMING
    PINMODE3 = 0x00C00000;              // 00000000 11000000 00000000 00000000
#endif

    PINSEL5 = 0x00000000;               // 00000000
                                        // xx         31 nc
                                        //   xx       30 nc
                                        //     xx     29 nc
                                        //       xx   28 nc
                                        // 00000000
                                        // xx         27 nc
                                        //   xx       26 nc
                                        //     xx     25 nc
                                        //       xx   24 nc
                                        // 00000000
                                        // xx         23 nc
                                        //   xx       22 nc
                                        //     xx     21 nc
                                        //       xx   20 nc
                                        // 00000000
                                        // xx         19 nc
                                        //   xx       18 nc
                                        //     xx     17 nc
                                        //       xx   16 nc

    PINSEL4 = 0x0A800000;               // 00001010
                                        // xx         15 nc
                                        //   xx       14 nc
                                        //     xx     13 mmc_dat3
                                        //       xx   12 mmc_dat2
                                        // 10000000
                                        // xx         11 mmc_dat1
                                        //   xx       10 kb mux c
                                        //     xx     09 kb mux b
                                        //       xx   08 kb mux a
                                        // 00000000
                                        // xx         07 led on
                                        //   xx       06 lcd6
                                        //     xx     05 lcd5
                                        //       xx   04 lcd4
                                        // 00000000
                                        // xx         03 lcd3
                                        //   xx       02 lcd2
                                        //     xx     01 lcd1
                                        //       xx   00 lcd0


#if RSR903
    FIO2SET = 0x000003FF;               //
    FIO2CLR = 0xFFFFFC00;               //
#else
    FIO2SET = 0x00000000;               //
    FIO2CLR = 0xFFFFFFFF;               //
#endif
    FIO2DIR = 0x000007FF;               // 00000000 00000000 00000111 11111111



    PINSEL7 = 0x000C0000;               // 00000000 00000000 00000000 00000000
                                        // xx         31 nc
                                        //   xx       30 nc
                                        //     xx     29 nc
                                        //       xx   28 nc
                                        // 00001100
                                        // xx         27 nc
                                        //   xx       26 gpio MDB/MICROMECH
                                        //     xx     25 pwm1.2 buzzer
                                        //       xx   24 nc
                                        // 00000000
                                        // xx         23 nc
                                        //   xx       22 nc
                                        //     xx     21 nc
                                        //       xx   20 nc
                                        // 00000000
                                        // xx         19 nc
                                        //   xx       18 nc
                                        //     xx     17 nc
                                        //       xx   16 nc

    PINSEL6 = 0x00000000;               // 00000000 00000000 00000000 00000000

    FIO3DIR = 0x02000000;               // 00000010 00000000 00000000 00000000 
    FIO3SET = 0x00000000;               // gpio3.26 = 0, Micromech
    FIO3CLR = 0xFFFFFFFF;               //


#if ANTI_V
    PINSEL9 = 0x00000000;               // 00000000
                                        // xx         31 nc
                                        //   xx       30 nc
                                        //     xx     29 i/o
                                        //       xx   28 i/o
                                        // 00000000
                                        // xx         27 nc
                                        //   xx       26 nc
                                        //     xx     25 nc
                                        //       xx   24 nc
                                        // 00000000
                                        // xx         23 nc
                                        //   xx       22 nc
                                        //     xx     21 nc
                                        //       xx   20 nc
                                        // 00000000
                                        // xx         19 nc
                                        //   xx       18 nc
                                        //     xx     17 nc
                                        //       xx   16 nc
    PINSEL8 = 0x00000000;               // 00000000 00000000 00000000 00000000

    FIO4DIR = 0x00000000;               // 00010000 00000000 00000000 00000000
#else
    PINSEL9 = 0x0F000000;               // 00001111
                                        // xx         31 nc
                                        //   xx       30 nc
                                        //     xx     29 rxd3
                                        //       xx   28 txd3
                                        // 00000000
                                        // xx         27 nc
                                        //   xx       26 nc
                                        //     xx     25 nc
                                        //       xx   24 nc
                                        // 00000000
                                        // xx         23 nc
                                        //   xx       22 nc
                                        //     xx     21 nc
                                        //       xx   20 nc
                                        // 00000000
                                        // xx         19 nc
                                        //   xx       18 nc
                                        //     xx     17 nc
                                        //       xx   16 nc
    PINSEL8 = 0x00000000;               // 00000000 00000000 00000000 00000000

    FIO4DIR = 0x10000000;               // 00010000 00000000 00000000 00000000
#endif
    FIO4SET = 0x00000000;               //
    FIO4CLR = 0xFFFFFFFF;               //

//    PINSEL10= 0x00000000;               // 00000000 00000000 00000000 00000000


#endif
}                                               //  InitIOPorts




/*--------------------------------------------------------------------------
 | InitPwmTimer:inizializza timer per pwm
 |                  - pwm0 resetta pwm ogni ms e genera int
 |                  - pwm2 controlla contrasto lcd
 |                              --------------
 | In:
 | Out:
 +--------------------------------------------------------------------------*/

void            InitPwmTimer( void )
{
#ifndef __WIN32__
    PWM1TCR = 0X00;                     // 00000000  pwm disabled
    PWM1PR = 0;                         // no prescaler
    PWM1PC = 0;                         // zero prescaler
    PWM1TC = 0;                         // reset timer counter
    PWM1MR0 = (PERIFREQ /(long)F_PWM);  // set pwm0 at 4kHz (x caricabatteria)
    PWM1MR2 = PWM1MR0/2;                // reset pwm2
    PWM1PCR = 0x0000;                   // 00000000  pwm2 out disabled
                                        // 00000000
    PWM1LER = 0x05;                     // 00100101  pwm values loaded
    PWM1TCR = 0X09;                     // 00001001  pwm enabled
    PWM1IR = 0x0F0F;                    // reset int
    PWM1MCR = 0x00000002;               // 00000000
                                        // 00000000
                                        // 00000000
                                        // 00000010  reset on pwm0
#endif
}                                               //  InitPwmTimer




/*--------------------------------------------------------------------------
 | PWM_ISR:     interrupt service routine per PWM
 |                              --------------
 | In:
 | Out:
 +--------------------------------------------------------------------------*/

void            PWM_ISR( void )
{
#ifndef __WIN32__
    VICAddress = 0;                     // Clear interrupt in VIC.
#endif
}                                               //  PWM_ISR




/*--------------------------------------------------------------------------
 | InitInterrupt:inzializza i vettori d'interrupt delle periferiche
 |                              --------------
 | In:
 | Out:
 +--------------------------------------------------------------------------*/

void            InitInterrupt( void )
{
#ifndef __WIN32__
//    VIC_Init();  // azzera il vic
//    VIC_SetProtectionMode(UserandPrivilegedMode);  // vic variabile anche da user

    //  interrupt display
    VIC_SetVectoredIRQ(I2C_ISR,VIC_Slot9,VIC_I2C0);
    VIC_EnableInt(1<<VIC_I2C0);


    // Timer1 interrupt
//    VIC_SetVectoredIRQ(TIMER1_ISR,VIC_Slot4,VIC_TIMER1);
//    VIC_EnableInt(1<<VIC_TIMER1);


/*

    // RTC interrupt
    VIC_SetVectoredIRQ(RTC_ISR,VIC_Slot5,VIC_RTC);
    VIC_EnableInt(1<<VIC_RTC);
*/
#endif
}                                               //  InitInterrupt




/*--------------------------------------------------------------------------
 | initSPI0:    inizializzazione della SPI0
 |                              --------------
 | In:
 | Out:
 +--------------------------------------------------------------------------*/

void            initSPI0( void )
{
#ifndef __WIN32__
    DWORD       dummy;

    S0SPCCR = 0x20;//0x8;                      // Setting SPI0 clock, 12MHz / 8 = 1,5Mhz
    dummy = S0SPSR;
    dummy = S0SPDR;                     // clear flags
    S0SPCR = 0x38;
                                        // 00000000
                                        // xxxx       reserved
                                        //     xxxx   N/A bit transfer
                                        // 00111000
                                        // 0          SPIE=no interrupt
                                        //  0         LSBF=MSB first
                                        //   1        MSTR=MASTER
                                        //    1       CPOL=clock active low 
                                        //     1      CPHA=1 
                                        //      0     8 bit enable
                                        //       xx   reserved
#endif
}                                               //  initSPI0




/*--------------------------------------------------------------------------
 | sendSPI0:    trasmette un buffer sulla SPI0
 |                              --------------
 | In:
 | Out:
 +--------------------------------------------------------------------------*/

void            sendSPI0( byte *_ptr, byte _len )
{
#ifndef __WIN32__
    byte        i, j;
    
    PINSEL3 &= ~0x0003C300;
    IO1DIR  |=  0x01100000;             // 00000001 00010000 00000000 00000000
    IO1DIR  &= ~0x00800000;             // 00000000 80000000 00000000 00000000
    while ( _len-- != 0 )
    {
        j = *_ptr++;
        for ( i = 0; i < 8; i++ )
        {
            SPI0_CLK_LOW();
            if ( (j&0x80 ) )
                SPI0_MOSI_HIGH();
            else
                SPI0_MOSI_LOW();
            SPI0_CLK_HIGH();
            j <<= 1;
        }
    }
#endif
}                                               //  sendSPI0




/*--------------------------------------------------------------------------
 | recvSPI0:    trasmette un buffer sulla SPI0
 |                              --------------
 | In:
 | Out:
 +--------------------------------------------------------------------------*/

void            recvSPI0( byte *_ptr, byte _len )
{
#ifndef __WIN32__
#if 0
    dword       dummy;

    while ( _len-- != 0 )
    {
        dummy =  S0SPSR;
		S0SPDR = *_ptr;
		while ( !(S0SPSR_bit.SPIF) );
		dummy = S0SPDR;		// Flush the RxFIFO
		_ptr++;
    }

#else
    byte        i;
    
//    PINSEL3 &= ~0x0003C300;
//    IO1DIR  |=  0x00100000;             // 00000000 00010000 00000000 00000000
    IO1DIR  &= ~0x01800000;             // 00000001 80000000 00000000 00000000
    while ( _len-- != 0 )
    {
        *_ptr = 0;
        for ( i = 0; i < 8; i++ )
        {
            *_ptr <<= 1;
            SPI0_CLK_LOW();
            if ( SPI0_MISO() )
                *_ptr |= 0x01;
            SPI0_CLK_HIGH();
        }
        *_ptr++;
    }

#endif
#endif
}                                               //  recvSPI0




/*--------------------------------------------------------------------------
 | flushSSP0:   svuota SSP0
 |                              --------------
 | In:
 | Out:
 +--------------------------------------------------------------------------*/

#define FIFOSIZE    8

void            flushSSP0( void )
{
#ifndef __WIN32__
    word        i, dummy;
    for ( i = 0; i < FIFOSIZE; i++ )
		dummy = SSP0DR;		/* clear the RxFIFO */
#endif
}                                               //  flushSSP0




/*--------------------------------------------------------------------------
 | initSSP0:    inizializzazione della SSP0
 |                              --------------
 | In:
 | Out:
 +--------------------------------------------------------------------------*/

#define FIFOSIZE    8

void            initSSP0( void )
{
#ifndef __WIN32__
    word        i, dummy;

	PCONP |= BIT21;                     // by default, it's enabled already, for safety reason

    /* Set DSS data to 8-bit, Frame format SPI, CPOL = 0, CPHA = 0, and SCR is 7 */
/*slow*/    SSP0CR0 = 0x0747;
//medium    SSP0CR0 = 0x0547;
//fast    SSP0CR0 = 0x0247;

    /* SSPCPSR clock prescale register, master mode, minimum divisor is 0x02 */
//fast    SSP0CPSR = 0x02;
/*slow*/    SSP0CPSR = 0x04;

    for ( i = 0; i < FIFOSIZE; i++ )
		dummy = SSP0DR;		/* clear the RxFIFO */

	/* Device select as master, SSP Enabled, loopback operational mode */
    SSP0CR1 = BIT1;//SSPCR1_LBM | SSPCR1_SSE;
    /* Set SSPINMS registers to enable interrupts */
    /* enable all error related interrupts */
    SSP0IMSC = 0;//SSPIMSC_RORIM | SSPIMSC_RTIM;
#endif
}                                               //  initSSP0





/*--------------------------------------------------------------------------
 | sendSSP0:    trasmette un dato sulla SPI0
 |                              --------------
 | In:  _data   dato da trasmettere
 | Out:
 +--------------------------------------------------------------------------*/

word            sendSSP0( byte *_ptr, word _len )
{
#ifndef __WIN32__
    word        i;
    
///    SSP0CR1 = BIT1;//SSPCR1_SSE;
    for ( i = 0 ; i < _len; i++ )
    {
		/* as long as TNF bit is set (TxFIFO is not full), I can always transmit */
		while ( !(SSP0SR & BIT1/*SSPSR_TNF*/) );
		SSP0DR = *_ptr++;
//		/* Wait until the Busy bit is cleared */
//		while ( (SSP0SR & BIT4/*SSPSR_BSY*/) );
    }
///    while ( (SSP0SR & BIT4/*SSPSR_BSY*/) );
///    SSP0CR1 = 0;

    return i;
#endif
}                                               //  sendSSP0

word            sendRecvSSP0( byte *_ptrW, byte *_ptrR, word _len )
{
#ifndef __WIN32__
    word        i;
    
///    SSP0CR1 = BIT1;//SSPCR1_SSE;
    for ( i = 0 ; i < _len; i++ )
    {
		/* as long as TNF bit is set (TxFIFO is not full), I can always transmit */
		while ( !(SSP0SR & BIT1/*SSPSR_TNF*/) );
		SSP0DR = *_ptrW++;
		/* Wait until the Busy bit is cleared */
//		while ( !(SSP0SR & BIT4/*SSPSR_BSY*/) );
        while ( !(SSP0SR & BIT2/*SSPSR_RNE*/) );
        *_ptrR++ = SSP0DR;
    }
///    while ( (SSP0SR & BIT4/*SSPSR_BSY*/) );
///    SSP0CR1 = 0;

    return i;
#endif
}



/*--------------------------------------------------------------------------
 | recvSSP0:    legge i dati letti dalla SPI0
 |                              --------------
 | In:  _ptr    puntatore al buffer dove porre i dati letti
 |      _len    numero di dati da leggere
 | Out:
 +--------------------------------------------------------------------------*/

word            recvSSP0( byte *_ptr, word _len )
{
#ifndef __WIN32__
    word        i;

    for ( i = 0; i <  _len; i++ )
    {
		/* As long as Receive FIFO is not empty, I can always receive. */
		/* since it's a loopback test, clock is shared for both TX and RX,
		no need to write dummy byte to get clock to get the data */
		/* if it's a peer-to-peer communication, SSPDR needs to be written
		before a read can take place. */
//		SSP0DR = 0xFF;
//		while ( !(SSP0SR & BIT2/*SSPSR_RNE*/) );
		if ( !(SSP0SR & BIT2/*SSPSR_RNE*/) )
            break;//empty
		*_ptr++ = SSP0DR;
    }
    return i;
#endif
}                                               //  recvSSP0




/*--------------------------------------------------------------------------
 | sendDate:    memorizza la data e l'ora
 |              (l'anno viene memorizzato nell'area per gli allarmi)
 |                              --------------
 | In:  dateTime    puntatore alla struttura con l'ora da scrivere
 | Out:
 +--------------------------------------------------------------------------*/

void            sendDate( _dateTime *dateTime )
{
#ifndef __WIN32__
 #if !RSR903
    dateTime -> year %= 100;
    bufLcdIIC[0] = 0xD0;
    bufLcdIIC[1] = 0x00;
    bufLcdIIC[2] = 0x80;                // stop orologio
    while(WriteI2c0(bufLcdIIC,3));

    bufLcdIIC[0] = 0xD0;
    bufLcdIIC[1] = 0x00;
    bufLcdIIC[2] = 0x80 | ((dateTime->sec   / 10) << 4) + (dateTime->sec   % 10);
    bufLcdIIC[3] =        ((dateTime->min   / 10) << 4) + (dateTime->min   % 10);
    bufLcdIIC[4] =        ((dateTime->hour  / 10) << 4) + (dateTime->hour  % 10);
    bufLcdIIC[5] = 0x00;
    bufLcdIIC[6] =        ((dateTime->day   / 10) << 4) + (dateTime->day   % 10);
    bufLcdIIC[7] =        ((dateTime->month / 10) << 4) + (dateTime->month % 10);
    bufLcdIIC[8] =        ((dateTime->year  / 10) << 4) + (dateTime->year  % 10);
    while(WriteI2c0(bufLcdIIC,9));

    bufLcdIIC[0] = 0xD0;
    bufLcdIIC[1] = 0x00;
    bufLcdIIC[2] = ((dateTime->sec / 10) << 4) + (dateTime->sec % 10);  // start orologio
    while(WriteI2c0(bufLcdIIC,3));
 #endif
#endif
}                                               //  sendDate

/*--------------------------------------------------------------------------
 | recvDate:    legge la data e l'ora
 |                              --------------
 | In:  dateTime    puntatore alla struttura con l'ora
 | Out: byte    0   lettura completata con successo
 |              1   errore nella lettura
 +--------------------------------------------------------------------------*/
byte            readClock( _dateTime *dateTime, byte addr );

byte            recvDate( _dateTime *dateTime )
{
    return( readClock( dateTime, 0x00 ));
}

DWORD   testTimeOff( void )
{
    _dateTime temp;
    DWORD	minSinceBigBang;
    
    if( readClock( &temp, 0x08 ))
        return( 0xFFFFFFFFL );
    minSinceBigBang = ( SinceBigBang( temp.day, temp.month, temp.year )*24L + temp.hour ) * 60L + temp.min;
    
    if( readClock( &temp, 0x00 ))
        return( 0xFFFFFFFFL );
    return( ( SinceBigBang( temp.day, temp.month, temp.year )*24L + temp.hour ) * 60L + temp.min - minSinceBigBang + 1);
}

byte            readClock( _dateTime *dateTime, byte addr )
{
#ifdef __WIN32__
    time_t      t;
    struct tm   *tblock;

    t = time(NULL);
    tblock = localtime(&t);
    dateTime->hour = tblock->tm_hour;
    dateTime->min = tblock->tm_min;
    dateTime->sec = tblock->tm_sec;
    dateTime->day = tblock->tm_mday;
    dateTime->month = tblock->tm_mon+1;
    dateTime->year = tblock->tm_year+1900;
#else
 #if !RSR903
    byte        i, error;
    _dateTime   temp;

    i = 0;
    for ( i = 0,error = 0; i < 3 && error < 3; i++ )
    {
        bufLcdIIC[0] = 0xD0;
        bufLcdIIC[1] = addr;
        while(WriteI2c0(bufLcdIIC,2));  // set address

        bufLcdIIC[1] = 0xD1;
        while(ReadI2c0(&bufLcdIIC[1],10));   // read data

        temp.sec   = ((bufLcdIIC[2] & 0x70) >> 4) * 10 + (bufLcdIIC[2] & 0x0F);
        temp.min   = ((bufLcdIIC[3] & 0x70) >> 4) * 10 + (bufLcdIIC[3] & 0x0F);
        temp.hour  = ((bufLcdIIC[4] & 0x30) >> 4) * 10 + (bufLcdIIC[4] & 0x0F);
        temp.day   = ((bufLcdIIC[6] & 0x30) >> 4) * 10 + (bufLcdIIC[6] & 0x0F);
        temp.month = ((bufLcdIIC[7] & 0x10) >> 4) * 10 + (bufLcdIIC[7] & 0x0F);
        temp.year  = ((bufLcdIIC[8] & 0xF0) >> 4) * 10 + (bufLcdIIC[8] & 0x0F) + 2000;
        if ( temp.sec < 60 && temp.min < 60 && temp.hour < 24 && temp.day < 32 && temp.month < 13 )
        {
            memcpy( (char*)dateTime, (char*)&temp, sizeof(temp) );
            if( addr == 0 && bufLcdIIC[2] != bufLcdIIC[10] )
            {
                bufLcdIIC[0] = 0xD0;
                bufLcdIIC[1] = 0x08;
                while(WriteI2c0(bufLcdIIC,9));  // set address
            }
            return 0;
        }
        else
            error++;
    }
 #endif
#endif
    return 1;
}                                               //  recvDate




unsigned long command[5];
unsigned long result[2];

#define IAP_LOCATION         0x7FFFFFF1
#define SIZE_LOW_SECTOR      0x1000
#define SIZE_HIGH_SECTOR     0x8000
#define MIN_SIZE_BUFFER      256

#define PREPARE_SECTOR       50
#define COPY_RAM_TO_FLASH    51
#define ERASE_SECTOR         52
#define BLANK_CHECK          53
#define READ_ID              54
#define BOOT_VERSION         55
#define COMPARE              56
#define REINVOKE_ISP         57

enum FLASHRESULT{
  CMD_SUCCESS = 0,                                    // Command is executed successfully.
  INVALID_COMMAND,                                    // Invalid command.
  SRC_ADDR_ERROR,                                     // Source address is not on a word boundary.
  DST_ADDR_ERROR,                                     // Destination address is not on a correct boundary.
  SRC_ADDR_NOT_MAPPED,                                // Source address is not mapped in the memory map.
                                                      // Count value is taken in to consideration where applicable.
  DST_ADDR_NOT_MAPPED,                                // Destination address is not mapped in the memory map.
                                                      // Count value is taken in to consideration where applicable.
  COUNT_ERROR,                                        // Byte count is not multiple of 4 or is not a permitted value.
  INVALID_SECTOR,                                     // Sector number is invalid.
  SECTOR_NOT_BLANK,                                   // Sector is not blank.
  SECTOR_NOT_PREPARED_FOR_WRITE_OPERATION,            // Command to prepare sector for write operation was not executed.
  COMPARE_ERROR,                                      // Source and destination data is not same.
  BUSY                                                // Flash programming hardware interface is busy.
                };

typedef void (*IAP)(unsigned long *, unsigned long *);




/***************************************************************************
 GetBlock:      ritorna il numero del blocco corrispondente all'indirizzo
								--------------
 In:	_addr   indirizzo di inizio del blocco
 Out:   numero blocco
 ***************************/

byte            GetBlock( dword _address )
{
    if ( _address < 0x0008000L )
        return (_address / SIZE_LOW_SECTOR);
    if ( _address >= 0x0078000L )
        return 22+((_address-0x78000L) / SIZE_LOW_SECTOR);
    return (8 + ((_address - (SIZE_LOW_SECTOR * 8)) / SIZE_HIGH_SECTOR));
}                                               //  GetBlock




/***************************************************************************
 WriteFlash256: scrive 256 byte nella flash interna al micro
								--------------
 In:	_buf        puntatore al buffer da memorizzare (word boundary)
        _address    indirizzo della flash in cui scrivere (divisibile per 256)
 Out:   0       ok
        1       ko
 ***************************/

byte            WriteFlash256( void *buf, dword _address, word _size )
{
#ifdef  __WIN32__
    return 0;
#else
    byte        nBlock;
    DWORD       ramBuf[256/sizeof(DWORD)];     

    if ( _size == 0 || _size > sizeof(ramBuf) )
        memcpy( ramBuf, buf, sizeof(ramBuf) );
    else
    {
        memset( ramBuf, 0, sizeof(ramBuf) );
        memcpy( ramBuf, buf, _size );
    }

    IAP iap_entry = (IAP) IAP_LOCATION;
    nBlock = GetBlock(_address);
    command[0] = PREPARE_SECTOR;
    command[1] = nBlock;
    command[2] = command[1];
    result[0] = 0xFF;    
    __disable_interrupt();
    iap_entry(command, result);
    __enable_interrupt();
    if ( result[0] != CMD_SUCCESS )
        return 1;
    command[0] = COPY_RAM_TO_FLASH;
    command[1] = _address;
    command[2] = (unsigned long)ramBuf;
    command[3] = 256;
    command[4] = SYSTEMFREQ / 1000;
    result[0] = 0xFF;   
    __disable_interrupt();
#ifdef BOOT2  
    MEMMAP_bit.MAP = USER_FLASH;
#endif    
    iap_entry(command, result);
#ifdef BOOT2    
    MEMMAP_bit.MAP = USER_RAM;
#endif     
    __enable_interrupt();   
    if ( result[0] != CMD_SUCCESS )
      return 2;

    return 0;
#endif
}                                               //  WriteFlash256




/***************************************************************************
 EraseBlock:    cancella un blocco della flash
								--------------
 In:	_addr   indirizzo di inizio del blocco da cancellare
        _num    numero blocchi da cancellare
 Out:   0       tutto OK
        1       preparazione settori impossibile
        2       cancellazione errata
 ***************************/

byte            EraseBlock( dword _address, byte _num )
{
#ifdef  __WIN32__
    return 0;
#else
    byte        nBlock;

    dword       *add;
    dword       size;

    MDBsafeMode();
    
                                        // verifica se la memoria e' gia' cancellata
    add = (dword *)_address;
    if ( _address < 0x0008000L || _address >= 0x0078000L )
        size = _num*0x1000L;
    else
        size = _num*0x8000L;
    while( size > 0 )
    {
        if ( *add != 0xFFFFFFFF )
            break;
        add++;
        size -= sizeof(dword);
    }
    if ( size == 0 )
        return 0;

                                        // prepara i settori alla cancellazione
    IAP iap_entry = (IAP) IAP_LOCATION;
    nBlock = GetBlock(_address);
    command[0] = PREPARE_SECTOR;
    command[1] = nBlock;
    command[2] = command[1]+(_num-1);
    result[0] = 0xFF;     
    __disable_interrupt();
    iap_entry(command, result);
    __enable_interrupt();
    if ( result[0] != CMD_SUCCESS )
        return 1;
                                        // cancella i settori
    command[0] = ERASE_SECTOR;
    command[1] = nBlock;
    command[2] = command[1]+(_num-1);
    command[3] = SYSTEMFREQ / 1000;
    result[0] = 0xFF;       
    __disable_interrupt();
#ifdef BOOT2  
    MEMMAP_bit.MAP = USER_FLASH;
#endif     
    iap_entry(command, result);
#ifdef BOOT2    
    MEMMAP_bit.MAP = USER_RAM;
#endif     
    __enable_interrupt();  
    if ( result[0] != CMD_SUCCESS )
        return 2;

    return 0;
#endif
}                                               //  EraseBlock

void    rsr903c_detection( void )
{
//    PINSEL1 = 0x01652A81; 
//    IO0DIR  = 0x7C7885F0;               // 01111100 01111000 10000101 11110000
//    IO0CLR  = 0xF3FFFF7F;               // 11110011 11111111 11111111 01111111
//    IO0SET  = 0x00000080;               // 00000000 00000000 00000000 10000000    
//
//    PINSEL7 = 0x00000000;               
//    PINSEL6 = 0x00000000;               // 00000000 00000000 00000000 00000000
//    FIO3DIR = 0x02000000;               // 00000010 00000000 00000000 00000000
//    FIO3SET = 0x00000000;               // 
//    FIO3CLR = 0xFFFFFFFF;               //
    
    if (FIO3PIN_bit.P3_26 == 0)
    {
        if (IO0PIN_bit.P0_23 == 1)
        {
            //rsr903c
            PINSEL1 = 0x01652A81;
            IO0CLR  = 0xF3FFFF5F;               // 11110011 11111111 11111111 01011111
            IO0SET  = 0x000000A0;               // 00000000 00000000 00000000 10100000  
            IO0DIR  = 0x7CF885F0;               // 01111100 11111000 10000101 11110000

            UCB_903C = RSR903C;
        }
        else {
            //error
            UCB_903C = 0xF1;
        }
    }
    else {
        if (IO0PIN_bit.P0_23 == 0)
        {
            PINSEL1 = 0x01656A81;                
            IO0CLR  = 0xF3FFFF5F;               // 11110011 11111111 11111111 01011111
            IO0SET  = 0x000000A0;               // 00000000 00000000 00000000 10100000
            IO0DIR  = 0x7C7885F0;               // 01111100 01111000 10000101 11110000
        
            PINSEL7 = 0x000C0000;        
            PINSEL6 = 0x00000000;               // 00000000 00000000 00000000 00000000        
            FIO3DIR = 0x06000000;               // 00000110 00000000 00000000 00000000
            FIO3SET = 0x00000000;               // gpio3.26 = 0, Micromech
            FIO3CLR = 0xFBFFFFFF;               //   
            //rsr641
            UCB_903C = UCB;
        }
        else {
            //error
            UCB_903C = 0xF2;
        }        
    }    
}                                               //  rsr903c_detection
