/*
 * @brief Make your board becomes a USB mouse
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2012
 * Copyright(C) Dean Camera, 2011, 2012
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "Mouse.h"
#include "Drivers\i2c.h"
#include "chip.h"      /* LPC11xx Peripheral Registers */

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

/** Buffer to hold the previously generated Mouse HID report, for comparison purposes inside the HID class driver. */
static uint8_t PrevMTouchHIDReportBuffer[sizeof(/*USB_MouseReport_Data_t*/USB_MTouchReport_Data_t)];

/** nxpUSBlib HID Class driver interface configuration and state information. This structure is
 *  passed to all HID Class driver functions, so that multiple instances of the same class
 *  within a device can be differentiated from one another.
 */
static USB_ClassInfo_HID_Device_t Mouse_HID_Interface = {
	.Config = {
		.InterfaceNumber              = 0,

		.ReportINEndpointNumber       = MOUSE_EPNUM,
		.ReportINEndpointSize         = MOUSE_EPSIZE,
		.ReportINEndpointDoubleBank   = false,

		.PrevReportINBuffer           = PrevMTouchHIDReportBuffer,
		.PrevReportINBufferSize       = sizeof(PrevMTouchHIDReportBuffer),
		.PortNumber             = 0,
	},
};

volatile uint8_t I2CMasterBuffer[BUFSIZE];
volatile uint8_t I2CSlaveBuffer[BUFSIZE];
volatile uint32_t I2CMasterState;
volatile uint32_t I2CReadLength, I2CWriteLength;

//The default I2C address(7-bit) of NCB30x1 would be 0x55
#define NCB_ADDR 0x55
typedef struct{
	uint8_t contact_id;
	uint8_t status;
	uint16_t x;
	uint16_t y;
	uint8_t area;
	uint8_t amplitude;
}MultiTouchMessage_t;

MultiTouchMessage_t MultiTouchMessage[10];
static uint8_t touchCount;

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Private functions
 ****************************************************************************/
#define LEDS_BAR		2
#define LED				20
#define TOUCH_INT		7
#define AUDIO_GAIN1		8
#define AUDIO_GAIN0		9

#define X_R_M         11
#define Y_R_P         12
#define X_R_P         13
#define Y_R_M         14
#define LIGHT_LCD_RETRO     16
#define MOTION         17
#define POWER_LCD      19
#define RESET          22
#define SHUTDOWN       23
#define MUTE          15

#define PORT0		0
#define PORT1		1

#define REG_NCB_RESET		0x0102
#define REG_NCB_ORIENT		0x010B
#define REG_NCB_IDLE		0x0108
#define REG_NCB_ACT			0x0109
#define REG_NCB_CALIBRATE	0x0120
#define REG_NCB_RESOLUTION	0x010C
#define REG_NCB_BACKUP		0x0103
#define NCB_TEST			0x01
#define NCB_NOTEST			0x00

/**
 * @brief Configures the board hardware and chip peripherals for the demo's functionality
 * @return Nothing
 */
static void SetupHardware(void)
{
	Board_Init();
	USB_Init(Mouse_HID_Interface.Config.PortNumber, USB_MODE_Device);
#if defined(USB_DEVICE_ROM_DRIVER)
	UsbdHid_Init();
#endif
}

/////////////////////////////////////////////////////////////////////
// HDMI
/////////////////////////////////////////////////////////////////////
const uint8_t  CONFIG[] = {
0X98,0XF4,0X80,                // Indirizzo CEC
0X98,0XF5,0X7C,                // Indirizzo INFOFRAME
0X98,0XF8,0X4C,                // Indirizzo DPLL
0X98,0XF9,0X64,                // Indirizzo KSV
0X98,0XFA,0X6C,                // Indirizzo EDID
0X98,0XFB,0X68,                // Indirizzo HDMI
0X98,0XFD,0X44,                // Indirizzo CP
0X44,0X6C,0X00,                // ADI Recomended setting
0X68,0X6F,0X0C,                // ADI recommended setting
0X68,0X85,0X1F,                // ADI recommended setting
0X68,0X87,0X70,                // ADI recommended setting
0X68,0X57,0XDA,                // ADI recommended setting
0X68,0X58,0X01,                // ADI recommended setting
0X68,0X03,0X98,                // ADI recommended setting
0X68,0X4C,0X44,                // ADI recommended setting
0X68,0XC1,0X01,                // ADI recommended setting
0X68,0XC2,0X01,                // ADI recommended setting
0X68,0XC3,0X01,                // ADI recommended setting
0X68,0XC4,0X01,                // ADI recommended setting
0X68,0XC5,0X01,                // ADI recommended setting
0X68,0XC6,0X01,                // ADI recommended setting
0X68,0XC7,0X01,                // ADI recommended setting
0X68,0XC8,0X01,                // ADI recommended setting
0X68,0XC9,0X01,                // ADI recommended setting
0X68,0XCA,0X01,                // ADI recommended setting
0X68,0XCB,0X01,                // ADI recommended setting
0X68,0XCC,0X01,                // ADI recommended setting
0X98,0X04,0X62,                // IO - OP_CH_SEL=RGB, XTAL=28.63636Mhz
0X98,0X06,0XA0,                // IO - Pos Polarity VS,HS, DE
0X98,0X0B,0X44,                // Power up part
0X98,0X0C,0X42,                // Power up part
0X98,0X14,0X7F,                // IO - Max Drive Strength
0X98,0X15,0X80,                // IO - Disable Tristate of Pins
0X80,0X2A,0X2F,                // CEC - Power UP CEC module
0X68,0X83,0XFE,                // HDMI - Enable Clock Terminator Port A
0X68,0X03,0X10,                // I2S mode 16 bit
0X68,0X6D,0X00,                // I2S CH1,2 SPIDIF0 TDM mode disabled
0X98,0X01,0X06,                // IO - PRIM_MODE=HDMI-GRAPHICS - VFREQ=60Hz
0X98,0X00,0X01,                // IO - VID_STD 800x600@60
0X98,0X03,0X40,                // IO - 24 bit SDR 444 Mode 0
0X44,0XC9,0X2D,                // CP - DIS_AUTO_PARAM_BUFF 0xC9[0]=1
0X98,0X02,0XF2,                // IO - Auto CSC, RGB out
0X98,0X19,0X83,                // IO - LLC DLL Phase
0X98,0X33,0X40,                // IO - LLC DLL enable
0X68,0X8D,0X04,                // LFG
0X68,0X8E,0X1E,                // HFG
0X68,0X1A,0X8A,                // unmute audio
0X68,0X57,0XDA,                // ADI recommended setting
0X68,0X58,0X01,                // ADI recommended setting
0X68,0X75,0X10,                // DDC drive strength
0X44,0XBA,0X03,                // CP - FREE RUN MODE 1
0X64,0X74,0X01                 // Enabled E-EDID
};

const uint8_t  EDID[256] = {
0x00,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x00,0x58,0xA9,
0x70,0x88,0x01,0x00,0x00,0x00,0x31,0x16,0x01,0x03,
0x90,0x0B,0x08,0x78,0x02,0x1E,0x55,0xA0,0x59,0x56,
0x9F,0x27,0x0D,0x50,0x54,0x00,0x00,0x00,0x45,0x40,
0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
0x01,0x01,0x01,0x01,0xA0,0x0F,0x20,0x00,0x31,0x58,
0x1C,0x20,0x28,0x80,0x14,0x00,0x20,0x58,0x32,0x00,
0x00,0x1E,0x00,0x00,0x00,0xFF,0x00,0x56,0x45,0x49,
0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
0x00,0x00,0x00,0xFC,0x00,0x52,0x53,0x52,0x38,0x38,
0x37,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x00,0x00,
0x00,0xFD,0x00,0x3A,0x3E,0x1E,0x53,0x03,0x02,0x0A,
0x20,0x20,0x20,0x20,0x20,0x20,0x01,0x6D,0x02,0x03,
0x11,0x40,0x41,0x01,0x23,0x09,0x02,0x40,0x66,0x03,
0x0C,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x01,0x04
};

void HDMI_Power( bool value );

/*
 * Inizializzazione HDMI
 */
void HDMI_Init(void)
{
  int i;

  I2CInit((uint32_t)I2CMASTER);

  for( i = 0; i < sizeof(CONFIG); i += 3){

// writes conf hdmi
    do{
      I2CWriteLength = 2 + 1;
      I2CReadLength = 0;
      I2CMasterBuffer[0] = CONFIG[i];
      I2CMasterBuffer[1] = CONFIG[i + 1];
      I2CMasterBuffer[2] = CONFIG[i + 2];
      I2CEngine();
    }while( I2C_CMD_OK != I2CMasterState);
// verify
 	  do{
      I2CWriteLength = 2;
   		I2CReadLength = 1;
	    I2CMasterBuffer[0] = CONFIG[i];
      I2CMasterBuffer[1] = CONFIG[i + 1];
	    I2CMasterBuffer[2] = CONFIG[i] | RD_BIT;
	    I2CSlaveBuffer[0] = ~CONFIG[i + 2];
	    I2CEngine();
 	  }while (I2C_CMD_OK != I2CMasterState);
 	  if(I2CSlaveBuffer[0] != CONFIG[i + 2]) i -= 3;  // if different redo
  }

// writes conf EDID
  for( i = 0; i < sizeof(EDID); i++ ){
    do{
      I2CWriteLength = 2 + 1;
      I2CReadLength = 0;
      I2CMasterBuffer[0] = 0X6C;
      I2CMasterBuffer[1] = i;
      I2CMasterBuffer[2] = EDID[i];
      I2CEngine();
    }while( I2C_CMD_OK != I2CMasterState);
// verify
    do{
      I2CWriteLength = 2;
  		I2CReadLength = 1;
	    I2CMasterBuffer[0] = 0X6C;
      I2CMasterBuffer[1] = i;
	    I2CMasterBuffer[2] = 0X6C | RD_BIT;
	    I2CSlaveBuffer[0] = ~EDID[i];
	    I2CEngine();
    }while (I2C_CMD_OK != I2CMasterState);
 	  if(I2CSlaveBuffer[0] != EDID[i]) i --;  // if different redo
  }
}

void IOInit(void)
{
	Chip_GPIO_SetDir( LPC_GPIO_PORT, PORT0, (1<<TOUCH_INT), 0 );
	Chip_GPIO_SetDir( LPC_GPIO_PORT, PORT0, (1<<LIGHT_LCD_RETRO), 1 );
	Chip_GPIO_SetDir( LPC_GPIO_PORT, PORT0, (1<<POWER_LCD), 1 );
	Chip_GPIO_SetDir( LPC_GPIO_PORT, PORT0, (1<<AUDIO_GAIN0), 1 );
	Chip_GPIO_SetDir( LPC_GPIO_PORT, PORT0, (1<<AUDIO_GAIN1), 1 );
	Chip_GPIO_SetDir( LPC_GPIO_PORT, PORT0, (1<<SHUTDOWN), 1 );
	Chip_GPIO_SetDir( LPC_GPIO_PORT, PORT1, (1<<MUTE), 1 );
	Chip_GPIO_SetDir( LPC_GPIO_PORT, PORT0, (1<<LED), 1 );
	Chip_GPIO_SetDir( LPC_GPIO_PORT, PORT0, (1<<RESET), 1 );
	Chip_GPIO_SetDir( LPC_GPIO_PORT, PORT0, (1<<LEDS_BAR), 1 );
	Chip_GPIO_SetDir( LPC_GPIO_PORT, PORT0, (1<<MOTION), 0 );

	Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, LIGHT_LCD_RETRO, 0 );
	Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, AUDIO_GAIN0, 1 );
	Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, AUDIO_GAIN1, 1 );
	Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, SHUTDOWN, 1 );
	Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT1, MUTE, 0 );
	Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, RESET, 0 );
	Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, LED, 1 );  // off
	Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, POWER_LCD, 1 );
	Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, LEDS_BAR, 0 );

	delayms( 100 );

	Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, LIGHT_LCD_RETRO, 1 );

}

/*
 * test communication with accelerometer
 */
void testAccelerometer( void )
{
	do{
		I2CWriteLength = 2;
		I2CReadLength = 1;
		I2CMasterBuffer[0] = 0X3C;
		I2CMasterBuffer[1] = 0x0F;
		I2CMasterBuffer[2] = 0X3C | RD_BIT;
		I2CEngine();
	}while (I2C_CMD_OK != I2CMasterState);

	if(I2CSlaveBuffer[0] != 0x49) {
		Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, LED, 0 );
	} else
		Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, LED, 1 );
}

/*
 * Delay 1 cycles = 1 ms
 */
void delayms( uint16_t cycles )
{
	uint16_t i;

	while( cycles )
	{
		cycles--;
		for( i = 0; i < 1750; i++ ) {}
	}
}

/**************************************************************************************
 * Function for special init hdmi
 **************************************************************************************/

void clr_write( uint8_t addr, uint8_t reg, uint8_t value )
{
	do{
		I2CWriteLength = 2;
		I2CReadLength = 1;
		I2CMasterBuffer[0] = addr;
		I2CMasterBuffer[1] = reg;
		I2CMasterBuffer[2] = addr | RD_BIT;
		I2CEngine();
	}while (I2C_CMD_OK != I2CMasterState);

	I2CSlaveBuffer[0] &= ~value;
    do{
      I2CWriteLength = 2 + 1;
      I2CReadLength = 0;
      I2CMasterBuffer[0] = addr;
      I2CMasterBuffer[1] = reg;
      I2CMasterBuffer[2] = I2CSlaveBuffer[0];
      I2CEngine();
    }while( I2C_CMD_OK != I2CMasterState);
}

void set_write( uint8_t addr, uint8_t reg, uint8_t value )
{
	do{
		I2CWriteLength = 2;
		I2CReadLength = 1;
		I2CMasterBuffer[0] = addr;
		I2CMasterBuffer[1] = reg;
		I2CMasterBuffer[2] = addr | RD_BIT;
		I2CEngine();
	}while (I2C_CMD_OK != I2CMasterState);

	I2CSlaveBuffer[0] |= value;
    do{
      I2CWriteLength = 2 + 1;
      I2CReadLength = 0;
      I2CMasterBuffer[0] = addr;
      I2CMasterBuffer[1] = reg;
      I2CMasterBuffer[2] = I2CSlaveBuffer[0];
      I2CEngine();
    }while( I2C_CMD_OK != I2CMasterState);
}

void HDMI_Power( bool value )
{
	if( value )
	{
		// power on
		clr_write( 0x98, 0x0C, 0x01 );
		//clr_write( 0x68, 0x73, 0xFF );
		/*clr_write( 0x98, 0x0C, 0x25 );
		clr_write( 0x98, 0x0B, 0x03 );
		clr_write( 0x80, 0x2A, 0x01 );
		clr_write( 0x68, 0x73, 0xFF );
		clr_write( 0x98, 0x15, 0x0E );*/
	} else {
		// power off
		// http://www.analog.com/static/imported-files/user_guides/UG-180.pdf pg.13 "POWER-DOWN CONTROLS"
		set_write( 0x98, 0x0C, 0x01 );
		//set_write( 0x68, 0x73, 0xFF );
		/*set_write( 0x98, 0x0C, 0x25 );
		set_write( 0x98, 0x0B, 0x03 );
		set_write( 0x80, 0x2A, 0x01 );
		set_write( 0x68, 0x73, 0xFF );
		set_write( 0x98, 0x15, 0x0E );*/
	}
}

/**************************************************************************************
 * Functions for NuTouch NCB3051 nuvoTon
 **************************************************************************************/

void ncb_writeRegs( uint16_t reg, uint8_t *data, uint8_t len, uint8_t test )
{
	uint8_t i;

	// write regs
	do{
		I2CWriteLength = 3 + len;
		I2CReadLength = 0;
		I2CMasterBuffer[0] = (NCB_ADDR<<1);
		I2CMasterBuffer[1] = (reg & 0x00FF);
		I2CMasterBuffer[2] = (reg >> 8);
		for( i = 0; i < len; i++ )
			I2CMasterBuffer[i+3] = data[i];
		I2CEngine();
	}while( I2C_CMD_OK != I2CMasterState);

	if( test )
	{
		// read regs wrote
		do{
			I2CWriteLength = 3;
			I2CReadLength = len;
			I2CMasterBuffer[0] = (NCB_ADDR<<1);
			I2CMasterBuffer[1] = (reg & 0x00FF);
			I2CMasterBuffer[2] = (reg >> 8);
			I2CMasterBuffer[3] = (NCB_ADDR<<1) | RD_BIT;
			I2CEngine();
		}while (I2C_CMD_OK != I2CMasterState);

		// test if wrote correctly
		for( i = 0; i < len; i++ )
		{
			if( data[i] != I2CSlaveBuffer[i] )
				while(1) {
					delayms(1000);
					Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, LEDS_BAR, 0 ); // led_bar off
					delayms(1000);
					Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, LEDS_BAR, 1 ); // led_bar on
				}
		}
	}
}

/*
 * Reset NCB
 */
void touchReset( void )
{
	uint8_t data = 0xFF;
	ncb_writeRegs( REG_NCB_RESET, &data, 1, NCB_NOTEST );
}

/*
 * Set InvertY InvertX Switch
 */
void touchOrient( uint8_t binverty, uint8_t binvertx, uint8_t bswitch)
{
	uint8_t mask;

	if( binvertx )
		binvertx = 0x01;

	if( binverty )
		binverty = 0x01;

	if( bswitch )
		bswitch = 0x01;

	mask = (binverty<<2) | (binvertx<<1) | bswitch;

	ncb_writeRegs( REG_NCB_ORIENT, &mask, 1, NCB_TEST );
}

/*
 * set Idle Acquisition Interval & Active Acquisition Interval
 */
void touchAcquisitionInterval( uint8_t idleinterval, uint8_t actinterval )
{
	uint8_t data[2];
	data[0] = idleinterval;
	data[1] = actinterval;
	ncb_writeRegs( REG_NCB_IDLE, data, 2, NCB_TEST );
}

/*
 * touch calibrate
 */
void touchCalibrate( void )
{
	uint8_t data = 0x20;
	ncb_writeRegs( REG_NCB_CALIBRATE, &data, 1, NCB_NOTEST );
}

/*
 * Set resolution
 */
void touchResolution( uint16_t x, uint16_t y )
{
	uint16_t data[2];

	data[0] = y & 0x0FFF;
	data[1] = x & 0x0FFF;

	ncb_writeRegs( REG_NCB_RESOLUTION, (uint8_t*)data, 4, NCB_TEST );
}

/*
 * L'NCB save configration
 */
void touchBackup( void )
{
	uint8_t data = 0x20;
	ncb_writeRegs( REG_NCB_BACKUP, &data, 1, NCB_NOTEST );
}

/*
 * reads coordinates from NCB3051
 */
int readCoord( void )
{
	if( ( Chip_GPIO_ReadPortBit( LPC_GPIO_PORT, PORT0, TOUCH_INT ) == 0 ) && ( touchCount == 0 ) )
	{
		uint8_t i;

		Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, LED, 0 ); // led on
		do{
			I2CWriteLength = 1;
			I2CReadLength = 60;
			I2CMasterBuffer[0] = (NCB_ADDR<<1) | RD_BIT;
			I2CEngine();
		}while (I2C_CMD_OK != I2CMasterState);
		Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, LED, 1 ); // led off

		//Read message of multi-touch report
		for(i=0;i<10;i++)
		{
			if(I2CSlaveBuffer[i*6] == 0xFF)
				break;

			MultiTouchMessage[i].contact_id = I2CSlaveBuffer[i*6]>>3;
			MultiTouchMessage[i].status = I2CSlaveBuffer[i*6]&0x07;
			MultiTouchMessage[i].x = (I2CSlaveBuffer[i*6+1] ) | ((I2CSlaveBuffer[i*6+3]&0x0F)<<8);
			MultiTouchMessage[i].y = (I2CSlaveBuffer[i*6+2] ) | ((I2CSlaveBuffer[i*6+3]&0xF0)<<4);
			MultiTouchMessage[i].area = I2CSlaveBuffer[i*6+4] ;
			MultiTouchMessage[i].amplitude = I2CSlaveBuffer[i*6+5];
			touchCount++;
		}
	}

	return touchCount;
}

/*****************************************************************************
 * Public functions
 ****************************************************************************/


#if defined(USB_DEVICE_ROM_DRIVER)
extern void CALLBACK_UsbdHid_SetReportChange(bool newstate);
#endif

void codeLed( uint8_t pulse )
{
	delayms(300);
	while( pulse )
	{
		pulse--;
		Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, LED, 0 ); // led on
		delayms(200);
		Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, LED, 1 ); // led off
		delayms(200);
	}
}

/**
 * @brief Main program entry point
 * @return Will not return
 * @note  This routine contains the overall program flow, including initial
 * setup of all components and the main program loop
 */
int main(void)
{
	LPC_IOCON->PIO0[POWER_LCD] = 0;	                  // power_lcd
	Chip_GPIO_SetDir( LPC_GPIO_PORT, PORT0, POWER_LCD, 1 );
	Chip_GPIO_WritePortBit( LPC_GPIO_PORT, PORT0, POWER_LCD, 0 );   // power LCD off

	SetupHardware();
	IOInit();

	codeLed(1);
	delayms( 10 );
	Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, RESET, 1 );
	delayms( 10 );
	HDMI_Init();

	codeLed(2);
	testAccelerometer();

	codeLed(3);
	touchReset();
	touchAcquisitionInterval( 32, 255 );
	touchResolution( 1279, 799 );
	touchOrient( 0, 1, 1 );
	touchCalibrate();

	Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, LEDS_BAR, 1 );

	for (;; ) {
		#if defined(USB_DEVICE_ROM_DRIVER)
		USB_MouseReport_Data_t report;
		uint16_t reportsize;
		uint8_t reportID = 0;

		memset(&report, 0, sizeof(USB_MouseReport_Data_t));
		CALLBACK_HID_Device_CreateHIDReport(&Mouse_HID_Interface, &reportID, HID_REPORT_ITEM_In, &report, &reportsize);
		if (memcmp(&report, Mouse_HID_Interface.Config.PrevReportINBuffer,
				   Mouse_HID_Interface.Config.PrevReportINBufferSize)) {
			memcpy(Mouse_HID_Interface.Config.PrevReportINBuffer,
				   &report,
				   Mouse_HID_Interface.Config.PrevReportINBufferSize);
			CALLBACK_UsbdHid_SetReportChange(true);
		}
		#else
		if( readCoord() )
		  HID_Device_USBTask(&Mouse_HID_Interface);
		else {
			if( Chip_GPIO_ReadPortBit( LPC_GPIO_PORT, PORT0, MOTION ) )
				Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, LED, 0 ); // led on
			else
				Chip_GPIO_WritePortBit(LPC_GPIO_PORT, PORT0, LED, 1 ); // led off
		}
		USB_USBTask(Mouse_HID_Interface.Config.PortNumber, USB_MODE_Device);
		#endif
	}
}

/**
 * @brief Event handler for the library USB Connection event
 * @return Nothing
 */
void EVENT_USB_Device_Connect(void)
{
}


/**
 * @brief Event handler for the library USB Disconnection event
 * @return Nothing
 */
void EVENT_USB_Device_Disconnect(void)
{
}

/**
 * @brief Event handler for the library USB Configuration Changed event
 * @return Nothing
 */
void EVENT_USB_Device_ConfigurationChanged(void)
{
	bool ConfigSuccess = true;

	ConfigSuccess &= HID_Device_ConfigureEndpoints(&Mouse_HID_Interface);

	USB_Device_EnableSOFEvents();
}

/**
 * @brief Event handler for the library USB Control Request reception event
 * @return Nothing
 */
void EVENT_USB_Device_ControlRequest(void)
{
	HID_Device_ProcessControlRequest(&Mouse_HID_Interface);
}

/**
 * @brief Event handler for the USB device Start Of Frame event
 * @return Nothing
 */
void EVENT_USB_Device_StartOfFrame(void)
{
	HID_Device_MillisecondElapsed(&Mouse_HID_Interface);
}

/* HID class driver callback function for the creation of HID reports to the host */
bool CALLBACK_HID_Device_CreateHIDReport(USB_ClassInfo_HID_Device_t *const HIDInterfaceInfo,
										 uint8_t *const ReportID,
										 const uint8_t ReportType,
										 void *ReportData,
										 uint16_t *const ReportSize)
{
	USB_MTouchReport_Data_t *MouseReport = (USB_MTouchReport_Data_t *) ReportData;
	bool press = false;

	// Request number touch points
	if( *ReportID == 0x04 && ReportType == 0x02  )
	{
		uint8_t *frame = (uint8_t *) ReportData;
		frame[0] = 0x04;
		frame[1] = 0x0A;
		*ReportSize = 2;
		return true;
	}
	else if( touchCount )
	{
		touchCount--;
		float x = MultiTouchMessage[touchCount].x * 25;
		float y = MultiTouchMessage[touchCount].y * 40;
		MouseReport->report_id = 0x04;
		MouseReport->usages = 0x80 | ((MultiTouchMessage[touchCount].contact_id-1)<<2);
		if( MultiTouchMessage[touchCount].status == 0x01 || MultiTouchMessage[touchCount].status == 0x03 )
			MouseReport->usages |= 0x03;
		else
			MouseReport->usages &= 0xFE;

		MouseReport->x = x;
		MouseReport->y = y;
		press = true;
	} else {
		press = false;
		*ReportSize = 0;
		return press;
	}

	*ReportSize = sizeof(USB_MTouchReport_Data_t);
	return press;
}

/* HID class driver callback function for the processing of HID reports from the host */
void CALLBACK_HID_Device_ProcessHIDReport(USB_ClassInfo_HID_Device_t *const HIDInterfaceInfo,
										  const uint8_t ReportID,
										  const uint8_t ReportType,
										  const void *ReportData,
										  const uint16_t ReportSize)
{
	/* Unused (but mandatory for the HID class driver) in this demo, since there are no Host->Device reports */
}
